-- DCS Infantry Attack Script using MOOSE
-- Manages InfantryTeamAlpha for automatic enemy engagement / return & embark
-----------------------------------------------------------------------
-- Configuration
-----------------------------------------------------------------------
local INFANTRY_GROUP_NAME   = "InfantryTeamAlpha"
local PLAYER_GROUP_NAME     = "Player-chinook"
local ENEMY_SEARCH_DISTANCE = 5000 -- meters
local CHECK_INTERVAL        = 5    -- seconds (status)
local ENEMY_CHECK_INTERVAL  = 3    -- seconds (elimination)

-----------------------------------------------------------------------
-- Globals
-----------------------------------------------------------------------
local infantryMooseGroup     = nil   -- MOOSE GROUP
local playerMooseGroup       = nil   -- MOOSE GROUP (transport helo)
local currentTargetGroups    = {}    -- array of enemy MOOSE GROUPs
local isDeployed             = false
local enemyCheckScheduler    = nil   -- timer handle (number)
local debugEnabled           = true

-----------------------------------------------------------------------
-- Debug message
-----------------------------------------------------------------------
local function debugMsg(message)
  if debugEnabled then
    MESSAGE:New("INFANTRY DEBUG: " .. message, 10):ToAll()
    BASE:E("DEBUG [Infantry]: " .. message)
  end
end

-----------------------------------------------------------------------
--  Make infantry aggressive: prefer MOOSE wrappers, fallback to DCS controller:setOption
-----------------------------------------------------------------------

local function setInfantryAggressive(g)
  if not g or not g:IsAlive() then
    debugMsg("setInfantryAggressive: group invalid or dead")
    return
  end

  pcall(function() g:OptionROEWeaponFree() end)
  debugMsg("Set ROE to weapon free")
  -- pcall(function() g:OptionAlarmStateRed() end)
  -- debugMsg("Set alarm state to red")
  pcall(function() g:OptionAllowAbortMission(false) end)
  debugMsg("Set allow abort mission to false")
  pcall(function() g:OptionDisperseOnAttack(false) end)
  debugMsg("Set disperse on attack to false")
  pcall(function() g:OptionEngageRange(1000) end)
  debugMsg("Set engagement range to 1000m")

end


-----------------------------------------------------------------------
-- Initialize the script
-----------------------------------------------------------------------
local function initialize()
  debugMsg("Initializing infantry script with MOOSE...")

  infantryMooseGroup = GROUP:FindByName(INFANTRY_GROUP_NAME)
  if not infantryMooseGroup then
    debugMsg("ERROR: Infantry group '" .. INFANTRY_GROUP_NAME .. "' not found!")
    return false
  end

  playerMooseGroup = GROUP:FindByName(PLAYER_GROUP_NAME)
  if not playerMooseGroup then
    debugMsg("WARNING: Player group '" .. PLAYER_GROUP_NAME .. "' not found - return/embark may not work!")
  else
    debugMsg("Player group '" .. PLAYER_GROUP_NAME .. "' found successfully")
  end

  debugMsg("Infantry group '" .. INFANTRY_GROUP_NAME .. "' found successfully via MOOSE")
  return true
end

-- Finds enemy ground groups within rangeM of a friendly MOOSE GROUP.
-- Returns a list of { group = <MOOSE GROUP>, distance = <meters> }, sorted by distance asc.
local function findNearbyEnemyGroups(infantryGroup, rangeM)
  local results = {}
  if not infantryGroup or not infantryGroup:IsAlive() then
    return results
  end

  -- Determine enemy coalition string for MOOSE filtering.
  local friendlySide = infantryGroup:GetCoalition()
  local enemyCoalition =
    (friendlySide == coalition.side.BLUE and "red") or
    (friendlySide == coalition.side.RED  and "blue") or
    { "red", "blue" }

  -- Build a SET_GROUP of active enemy ground groups.
  local enemySet = SET_GROUP:New()
  enemySet:FilterCoalitions(enemyCoalition)
  enemySet
    :FilterCategoryGround()
    :FilterActive(true)
    :FilterStart()

  local infCoord = infantryGroup:GetCoordinate()

  -- Collect enemies within the specified range (meters), with extra filtering.
  enemySet:ForEachGroup(function(grp)
    if not grp or not grp:IsAlive() then return end
    if grp:GetName() == infantryGroup:GetName() then return end

    -- Ignore empty groups / statics / markers by heuristic.
    local unitCount = 0
    if grp.GetUnitsCount then
      unitCount = grp:GetUnitsCount()
    elseif grp.GetSize then
      unitCount = grp:GetSize()
    else
      -- Fallback: try enumerating units if available
      if grp.GetUnits then
        local tmpUnits = grp:GetUnits()
        if tmpUnits and type(tmpUnits) == "table" then
          unitCount = #tmpUnits
        end
      end
    end
    if unitCount == 0 then
      debugMsg("Ignoring '"..grp:GetName().."' (no units)")
      return
    end

    local gname = grp:GetName():lower()
    if string.find(gname, "bullseye") or string.find(gname, "marker") or string.find(gname, "trigger") then
      debugMsg("Ignoring '"..grp:GetName().."' (likely marker/bullseye)")
      return
    end

    local grpCoord = grp:GetCoordinate()
    if grpCoord then
      local dist = infCoord:Get2DDistance(grpCoord)
      if dist and dist <= rangeM then
        debugMsg(string.format("Enemy cand. '%s' at %.1f m", grp:GetName(), dist))
        table.insert(results, { group = grp, distance = dist })
      end
    end
  end)

  -- Sort by ascending distance.
  table.sort(results, function(a, b) return a.distance < b.distance end)
  debugMsg(string.format("Total enemy groups found in range: %d", #results))
  return results
end

-----------------------------------------------------------------------
-- Enemy elimination monitor (tracks groups, not units)
-----------------------------------------------------------------------
local function checkEnemiesEliminated()
  if not isDeployed or #currentTargetGroups == 0 then
    debugMsg("No active targets or not deployed - stopping enemy check")
    return
  end

  debugMsg("Checking if target enemy groups are eliminated...")

  local remaining = 0
  for _, g in ipairs(currentTargetGroups) do
    if g and g:IsAlive() then
      remaining = remaining + 1
    end
  end

  if remaining == 0 then
    debugMsg("All target enemy groups destroyed! Giving embark command")
    giveEmbarkCommand()  -- forward-declared below
  else
    debugMsg(string.format("Still %d of %d enemy groups remaining", remaining, #currentTargetGroups))
    enemyCheckScheduler = timer.scheduleFunction(checkEnemiesEliminated, nil, timer.getTime() + ENEMY_CHECK_INTERVAL)
  end
end

local function startEnemyCheck()
  if enemyCheckScheduler then
    timer.removeFunction(enemyCheckScheduler)
  end
  enemyCheckScheduler = timer.scheduleFunction(checkEnemiesEliminated, nil, timer.getTime() + ENEMY_CHECK_INTERVAL)
  debugMsg("Enemy elimination monitoring started")
end

-----------------------------------------------------------------------
-- Create multi-target route with attack commands at each waypoint
-----------------------------------------------------------------------
local function createMultiTargetRoute(enemyEntries)
  if not infantryMooseGroup or not infantryMooseGroup:IsAlive() then
    debugMsg("ERROR: Infantry group no longer exists")
    return
  end
  
  if #enemyEntries == 0 then
    debugMsg("No enemy entries to create route for")
    return
  end

  -- Get infantry starting position
  local infCoord = infantryMooseGroup:GetCoordinate()
  if not infCoord then
    debugMsg("ERROR: Could not get infantry position")
    return
  end

  -- Build waypoints array starting with current position
  local waypoints = {}
  
  -- Add starting waypoint
  table.insert(waypoints, {
    x = infCoord.x,
    y = infCoord.z,
    type = "Turning Point",
    action = "Off Road",
    speed = 15, -- km/h
    formation = "Off Road"
  })

  -- Add waypoint for each enemy group (already sorted by distance)
  for i, entry in ipairs(enemyEntries) do
    if entry.group and entry.group:IsAlive() then
      local targetCoord = entry.group:GetCoordinate()
      if targetCoord then
        local waypoint = {
          x = targetCoord.x,
          y = targetCoord.z,
          type = "Turning Point",
          action = "Off Road",
          speed = 15, -- km/h
          formation = "Off Road",
          task = {
            id = "ComboTask",
            params = {
              tasks = {
                {
                  id = "AttackGroup",
                  params = {
                    groupId = entry.group:GetID(),
                    weaponType = 805306368, -- Auto weapons
                    expend = "Auto",
                    attackQty = 1,
                    directionEnabled = false,
                    altitudeEnabled = false
                  }
                }
              }
            }
          }
        }
        
        table.insert(waypoints, waypoint)
        debugMsg(string.format("Added waypoint %d: Attack group '%s' at distance %.0fm", 
          i, entry.group:GetName(), entry.distance or 0))
      end
    end
  end

  -- Create the route
  local route = {
    id = 'Mission',
    params = {
      route = {
        points = waypoints
      }
    }
  }

  -- Get DCS group and set the route
  local dcsGroup = infantryMooseGroup:GetDCSObject()
  if dcsGroup then
    local controller = dcsGroup:getController()
    if controller then
      controller:setTask(route)
      debugMsg(string.format("Multi-target route created with %d waypoints", #waypoints))
      
      -- Build a flat list of GROUPs for tracking/monitoring
      local groupsOnly = {}
      for _, entry in ipairs(enemyEntries) do
        if entry.group and entry.group:IsAlive() then
          table.insert(groupsOnly, entry.group)
        end
      end
      
      currentTargetGroups = groupsOnly
      isDeployed = true
      
      startEnemyCheck()
      debugMsg(string.format("Started enemy elimination monitoring for %d target group(s)", #currentTargetGroups))
      
      return true
    else
      debugMsg("ERROR: Could not get group controller")
    end
  else
    debugMsg("ERROR: Could not get DCS group object")
  end
  
  return false
end

-----------------------------------------------------------------------
-- Move infantry to a target group (they'll engage automatically) - LEGACY SINGLE TARGET
-----------------------------------------------------------------------
local function moveToEnemyGroup(targetGroup, groupsInContact)
  if not infantryMooseGroup or not infantryMooseGroup:IsAlive() then
    debugMsg("ERROR: Infantry group no longer exists")
    return
  end
  if not targetGroup or not targetGroup:IsAlive() then
    debugMsg("ERROR: Target group invalid or dead")
    return
  end

  local targetCoord = targetGroup:GetCoordinate()
  if not targetCoord then
    debugMsg("ERROR: Could not get target group coordinate")
    return
  end

  debugMsg("Moving infantry to enemy group position - they will engage automatically")
  infantryMooseGroup:RouteGroundTo(targetCoord, 15) -- km/h

  local posStr = targetCoord:ToStringLLDMS()
  debugMsg(string.format("Infantry moving to target group '%s' at %s", targetGroup:GetName(), posStr))

  currentTargetGroups = groupsInContact or { targetGroup }
  isDeployed = true

  startEnemyCheck()
  debugMsg(string.format("Started enemy elimination monitoring for %d target group(s)", #currentTargetGroups))
end

-----------------------------------------------------------------------
-- Return infantry to helicopter and embark (single, correct task)
-----------------------------------------------------------------------
function returnToHelicopter()
  debugMsg("Return to helicopter command received")

  if not infantryMooseGroup or not infantryMooseGroup:IsAlive() then
    debugMsg("ERROR: Infantry group does not exist")
    return
  end

  if not playerMooseGroup or not playerMooseGroup:IsAlive() then
    debugMsg("ERROR: Player helicopter group not found or not alive")
    return
  end

  local playerCoord = playerMooseGroup:GetCoordinate()
  if not playerCoord then
    debugMsg("ERROR: Could not get player position")
    return
  end

  debugMsg(string.format("Player helicopter position: %s", playerCoord:ToStringLLDMS()))

  -- Use a single embark task that makes infantry route to the helo and wait.
  local radius = 200 -- meters
  local embarkTask = nil
  if infantryMooseGroup.TaskEmbarkToTransport then
    embarkTask = infantryMooseGroup:TaskEmbarkToTransport(playerCoord, radius)
  end

  if embarkTask then
    infantryMooseGroup:SetTask(embarkTask)
    debugMsg("Infantry given TaskEmbarkToTransport to player location (will move and wait)")
  else
    debugMsg("MOOSE embark task not available; attempting DCS fallback")
    local dcsGroup = infantryMooseGroup:GetDCSObject()
    if dcsGroup then
      local controller = dcsGroup:getController()
      if controller then
        controller:setTask({
          id = 'EmbarkToTransport',
          params = { x = playerCoord.x, y = playerCoord.z, zoneRadius = radius }
        })
        debugMsg("DCS fallback embark task set to player location")
      end
    end
  end

  -- clear internal state / monitoring
  isDeployed = false
  currentTargetGroups = {}

  if enemyCheckScheduler then
    timer.removeFunction(enemyCheckScheduler)
    enemyCheckScheduler = nil
    debugMsg("Stopped enemy elimination monitoring")
  end

  debugMsg("Infantry routing back to helicopter and will embark when they arrive")
end

-----------------------------------------------------------------------
-- Give embark command at current location (no move)
-----------------------------------------------------------------------
function giveEmbarkCommand()
  if not infantryMooseGroup or not infantryMooseGroup:IsAlive() then
    debugMsg("Cannot give embark command - infantry group doesn't exist")
    return
  end

  debugMsg("Giving embark command to infantry")

  -- Avoid canceling a route that should continue into embark; just set the embark task.
  local embarkTask = nil
  if infantryMooseGroup.TaskEmbarkToTransport then
    if playerMooseGroup and playerMooseGroup:IsAlive() then
      local pCoord = playerMooseGroup:GetCoordinate()
      if pCoord then
        embarkTask = infantryMooseGroup:TaskEmbarkToTransport(pCoord, 120)
        debugMsg("Embark task set to player position (MOOSE)")
      end
    end
    if not embarkTask then
      embarkTask = infantryMooseGroup:TaskEmbarkToTransport() -- embark 'here'
      debugMsg("Embark task set at infantry current position (MOOSE)")
    end
    infantryMooseGroup:SetTask(embarkTask)
  else
    debugMsg("TaskEmbarkToTransport not available, using DCS embark command...")
    local dcsGroup = infantryMooseGroup:GetDCSObject()
    if dcsGroup then
      local controller = dcsGroup:getController()
      if controller then
        local infantryCoord = infantryMooseGroup:GetCoordinate()
        controller:setTask({
          id = 'EmbarkToTransport',
          params = {
            x = infantryCoord.x,
            y = infantryCoord.z,
            zoneRadius = 100
          }
        })
        debugMsg("Embark task set using DCS controller")
      end
    end
  end

  isDeployed = false
  currentTargetGroups = {}

  if enemyCheckScheduler then
    timer.removeFunction(enemyCheckScheduler)
    enemyCheckScheduler = nil
    debugMsg("Stopped enemy elimination monitoring")
  end

  debugMsg("Infantry ready for embarkation")
end

-----------------------------------------------------------------------
-- Main attack function using MOOSE - UPDATED FOR MULTI-TARGET ROUTING
-----------------------------------------------------------------------
function attackCommand()
  debugMsg("Attack command received")

  if not infantryMooseGroup or not infantryMooseGroup:IsAlive() then
    debugMsg("ERROR: Infantry group does not exist")
    return
  end

  -- Set aggressive options right when attack is ordered
  setInfantryAggressive(infantryMooseGroup)

  local infCoord = infantryMooseGroup:GetCoordinate()
  if not infCoord then
    debugMsg("ERROR: Could not get infantry position")
    return
  end

  debugMsg(string.format("Infantry current position: %s", infCoord:ToStringLLDMS()))

  -- Find enemy groups within range (returns { {group=<GROUP>, distance=<m>}... } sorted by distance)
  local enemyEntries = findNearbyEnemyGroups(infantryMooseGroup, ENEMY_SEARCH_DISTANCE)

  if #enemyEntries > 0 then
    debugMsg(string.format("Found %d enemy group(s) within %dm - creating multi-target route", #enemyEntries, ENEMY_SEARCH_DISTANCE))

    -- Create multi-target route with attack commands at each waypoint
    if createMultiTargetRoute(enemyEntries) then
      debugMsg("Multi-target attack route created successfully")
    else
      debugMsg("Failed to create multi-target route, falling back to single target")
      -- Fallback to original single-target behavior
      local nearestEntry = enemyEntries[1]
      local nearestGroup = nearestEntry.group
      
      local groupsOnly = {}
      for _, entry in ipairs(enemyEntries) do
        if entry.group and entry.group:IsAlive() then
          table.insert(groupsOnly, entry.group)
        end
      end
      
      if nearestGroup and nearestGroup:IsAlive() then
        moveToEnemyGroup(nearestGroup, groupsOnly)
      else
        debugMsg("Nearest entry invalid or dead; giving embark instead")
        giveEmbarkCommand()
      end
    end
  else
    debugMsg(string.format("No enemies found within %dm - giving immediate embark command", ENEMY_SEARCH_DISTANCE))
    giveEmbarkCommand()
  end
end

-----------------------------------------------------------------------
-- Comms menu
-----------------------------------------------------------------------
local menuRoot = nil
local function addCommsMenu()
  menuRoot = MENU_MISSION:New("Infantry")
  MENU_MISSION_COMMAND:New("Attack",               menuRoot, attackCommand)
  MENU_MISSION_COMMAND:New("Return to Helicopter", menuRoot, returnToHelicopter)
  MENU_MISSION_COMMAND:New("Embark (Here)",        menuRoot, giveEmbarkCommand)
  debugMsg("Infantry commands added to F10 menu: Attack / Return / Embark")
end

-----------------------------------------------------------------------
-- Status monitor
-----------------------------------------------------------------------
local function monitorStatus()
  -- if infantryMooseGroup and infantryMooseGroup:IsAlive() then
  --   if isDeployed and #currentTargetGroups > 0 then
  --     debugMsg(string.format("Infantry status: Deployed, engaging %d target group(s)", #currentTargetGroups))
  --   else
  --     debugMsg("Infantry status: Ready for attack command")
  --   end
  -- else
  --   debugMsg("Infantry group not available")
  -- end

  -- timer.scheduleFunction(monitorStatus, nil, timer.getTime() + CHECK_INTERVAL)
end

-----------------------------------------------------------------------
-- Bootstrap
-----------------------------------------------------------------------
if initialize() then
  addCommsMenu()
  timer.scheduleFunction(monitorStatus, nil, timer.getTime() + CHECK_INTERVAL)
  debugMsg("DCS Infantry Attack Script with MOOSE loaded successfully!")
else
  debugMsg("Failed to initialize Infantry Attack Script!")
end